<?php

namespace App\Filament\Resources;

use App\Filament\Resources\NewsResource\Pages;
use App\Models\News;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;

use Illuminate\Support\Str;

class NewsResource extends Resource
{
    protected static ?string $model = News::class;

    protected static ?string $navigationIcon = 'heroicon-o-document-text';

    public static function getNavigationLabel(): string
    {
        return __('messages.news');
    }

    public static function getModelLabel(): string
    {
        return __('messages.news');
    }

    public static function getPluralModelLabel(): string
    {
        return __('messages.news');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Grid::make(3)
                    ->schema([
                        Forms\Components\Section::make()
                            ->schema([
                                Forms\Components\Select::make('category_id')
                                    ->label(__('التصنيف'))
                                    ->relationship('category', 'name_ar')
                                    ->required()
                                    ->searchable()
                                    ->preload(),
                                Forms\Components\TextInput::make('slug')
                                    ->label(__('الرابط'))
                                    ->required()
                                    ->maxLength(255)
                                    ->unique(News::class, 'slug', ignoreRecord: true),
                                Forms\Components\FileUpload::make('cover_image')
                                    ->label(__('صورة الغلاف'))
                                    ->image()
                                    ->directory('news'),
                                Forms\Components\Select::make('status')
                                    ->label(__('الحالة'))
                                    ->options([
                                        'draft' => __('مسودة'),
                                        'published' => __('منشور'),
                                    ])
                                    ->required()
                                    ->default('draft'),
                                Forms\Components\DateTimePicker::make('published_at')
                                    ->label(__('تاريخ النشر')),
                                Forms\Components\Toggle::make('is_pinned')
                                    ->label(__('تثبيت الخبر'))
                                    ->default(false),
                            ])->columnSpan(1),

                                Forms\Components\Tabs::make('Translations')
                            ->label(__('الترجمات'))
                            ->tabs([
                                Forms\Components\Tabs\Tab::make(__('العربية'))
                                    ->schema([
                                        Forms\Components\TextInput::make('title_ar')
                                            ->label(__('العنوان (عربي)'))
                                            ->required()
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (Forms\Set $set, ?string $state, Forms\Get $get) {
                                                if (!$get('slug')) {
                                                    $set('slug', Str::slug($state, language: 'ar'));
                                                }
                                            }),
                                        Forms\Components\MarkdownEditor::make('content_ar')
                                            ->label(__('المحتوى (عربي)'))
                                            ->required()
                                            ->columnSpanFull(),
                                    ]),
                                Forms\Components\Tabs\Tab::make(__('الإنجليزية'))
                                    ->schema([
                                        Forms\Components\TextInput::make('title_en')
                                            ->label(__('العنوان (إنجليزي)'))
                                            ->maxLength(255)
                                            ->live(onBlur: true)
                                            ->afterStateUpdated(function (Forms\Set $set, ?string $state, Forms\Get $get) {
                                                if (!$get('slug')) {
                                                    $set('slug', Str::slug($state));
                                                }
                                            }),
                                        Forms\Components\MarkdownEditor::make('content_en')
                                            ->label(__('المحتوى (إنجليزي)'))
                                            ->columnSpanFull(),
                                    ]),
                            ])->columnSpan(2),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ImageColumn::make('cover_image')
                    ->label(__('الصورة'))
                    ->circular(),
                Tables\Columns\TextColumn::make('title_ar')
                    ->label(__('العنوان (عربي)'))
                    ->searchable()
                    ->limit(50)
                    ->toggleable(),
                Tables\Columns\TextColumn::make('title_en')
                    ->label(__('العنوان (إنجليزي)'))
                    ->searchable()
                    ->limit(50)
                    ->toggleable(),
                Tables\Columns\TextColumn::make('category.name_ar')
                    ->label(__('التصنيف'))
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->label(__('الحالة'))
                    ->badge()
                    ->colors([
                        'warning' => 'draft',
                        'success' => 'published',
                    ]),
                Tables\Columns\TextColumn::make('published_at')
                    ->label(__('تاريخ النشر'))
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\IconColumn::make('is_pinned')
                    ->boolean()
                    ->label(__('مثبت')),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListNews::route('/'),
            'create' => Pages\CreateNews::route('/create'),
            'edit' => Pages\EditNews::route('/{record}/edit'),
        ];
    }
}
