<?php

namespace App\Http\Controllers;

use App\Models\FootballMatch;
use App\Models\League;
use App\Models\Player;
use App\Models\Team;
use App\Services\FootballApiService;
use Carbon\Carbon;
use Illuminate\Http\Request;

class FootballController extends Controller
{
    protected $api;

    public function __construct(FootballApiService $api)
    {
        $this->api = $api;
    }

    public function index()
    {
        $leagues = League::withCount('teams')->get();

        return view('football.index', compact('leagues'));
    }

    public function showLeague(League $league)
    {
        $teams = Team::where('league_id', $league->id)->get();
        $matches = FootballMatch::where('league_id', $league->id)
            ->orderBy('match_date', 'desc')
            ->take(10)
            ->get();

        return view('football.league', compact('league', 'teams', 'matches'));
    }

    public function showTeam(Team $team)
    {
        $players = Player::where('team_id', $team->id)->get();

        return view('football.team', compact('team', 'players'));
    }

    public function showMatch(FootballMatch $match)
    {
        return view('football.match', compact('match'));
    }

    public function matches(Request $request)
    {
        $date = $request->get('date', Carbon::today()->toDateString());
        $matches = FootballMatch::with('league')
            ->whereDate('match_date', $date)
            ->orderBy('match_date', 'asc')
            ->get();

        // Sort matches by league priority
        $leagueOrder = [233, 307, 39, 78, 140, 135]; // Egypt, Saudi, England, Germany, Spain, Italy

        $matches = $matches->sortBy(function ($match) use ($leagueOrder) {
            $index = array_search($match->league->api_league_id, $leagueOrder);
            return $index === false ? 999 : $index;
        });

        return view('football.matches', compact('matches', 'date'));
    }

    public function players()
    {
        $players = Player::with('team')->paginate(20);

        return view('football.players', compact('players'));
    }

    public function showPlayer(Player $player)
    {
        $player->load('team');

        // Fetch stats if empty or old (older than 1 day)
        if (empty($player->stats_json) || $player->updated_at->lt(now()->subDay())) {
            $season = 2025; // Default for 2025-2026 cycle
            $playerData = $this->api->getPlayer($player->api_player_id, $season);

            // Fallback to previous season if no data found for current season
            if (! $playerData || empty($playerData['statistics'])) {
                $playerData = $this->api->getPlayer($player->api_player_id, $season - 1);
            }

            if ($playerData) {
                $player->update([
                    'stats_json' => $playerData['statistics'] ?? [],
                ]);
            }
        }

        return view('football.player_profile', compact('player'));
    }

    public function standings(Request $request)
    {
        $leaguesToShow = [39, 140, 2, 12, 135, 78, 61, 233, 307, 3, 848]; // PL, La Liga, UCL, CAF CL, Serie A, Bundesliga, Ligue 1, Egypt, Saudi, UEL, UECL
        $leagues = League::whereIn('api_league_id', $leaguesToShow)
            ->where('is_enabled', true)
            ->orderByRaw('FIELD(api_league_id, '.implode(',', $leaguesToShow).')')
            ->get();

        return view('football.standings', compact('leagues'));
    }
}
