<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\FootballMatch;
use App\Models\League;
use App\Models\News;

class HomeController extends Controller
{
    public function index()
    {
        $leagueOrder = [233, 307, 39, 78, 140, 135, 203]; // Egypt, Saudi, England, Germany, Spain, Italy, Turkey
        $leaguesToShow = [39, 140, 2, 12, 135, 78, 61, 233, 307, 203, 3, 848]; // PL, La Liga, UCL, CAF CL, Serie A, Bundesliga, Ligue 1, Egypt, Saudi, Turkey, UEL, UECL
        $liveStatuses = ['1H', 'HT', '2H', 'ET', 'BT', 'P', 'LIVE'];

        $liveMatches = FootballMatch::where(function ($q) use ($liveStatuses) {
            $q->whereIn('status', $liveStatuses)
                ->orWhereRaw('status REGEXP "^[0-9]+$"')
                ->orWhere(function ($sq) {
                    $sq->where('status', 'FT')
                        ->where('live_visible_until', '>', now());
                });
        })->with('league')->get();

        // Sort live matches
        $liveMatches = $liveMatches->sortBy(function ($match) use ($leagueOrder) {
            $index = array_search($match->league->api_league_id, $leagueOrder);
            return $index === false ? 999 : $index;
        });

        $todayMatches = FootballMatch::whereDate('match_date', now()->toDateString())
            ->whereHas('league', function ($q) use ($leaguesToShow) {
                $q->whereIn('api_league_id', $leaguesToShow);
            })
            ->with('league')
            ->orderBy('match_date', 'asc')
            ->get();

        // Sort today matches
        $todayMatches = $todayMatches->sortBy(function ($match) use ($leagueOrder) {
            $index = array_search($match->league->api_league_id, $leagueOrder);
            return $index === false ? 999 : $index;
        });

        $yesterdayMatches = FootballMatch::whereDate('match_date', now()->subDay()->toDateString())
            ->whereHas('league', function ($q) use ($leaguesToShow) {
                $q->whereIn('api_league_id', $leaguesToShow);
            })
            ->with('league')
            ->orderBy('match_date', 'desc')
            ->get();

        // Sort yesterday matches
        $yesterdayMatches = $yesterdayMatches->sortBy(function ($match) use ($leagueOrder) {
            $index = array_search($match->league->api_league_id, $leagueOrder);
            return $index === false ? 999 : $index;
        });

        $upcomingMatches = FootballMatch::where('status', 'NS')
            ->where('match_date', '>=', now())
            ->whereHas('league', function ($q) {
                $q->whereIn('api_league_id', [233, 307, 39, 140, 135, 2, 12]);
            })
            ->with('league')
            ->orderBy('match_date', 'asc')
            ->take(5)
            ->get();
        $latestNews = News::where('status', 'published')
            ->orderBy('is_pinned', 'desc')
            ->orderBy('published_at', 'desc')
            ->take(6)
            ->get();
        $leagues = League::whereIn('api_league_id', $leaguesToShow)
            ->where('is_enabled', true)
            ->orderByRaw('FIELD(api_league_id, '.implode(',', $leaguesToShow).')')
            ->get();

        $standingsLeagues = League::whereIn('api_league_id', [39, 140, 135, 78, 61, 233, 307, 203])
            ->where('is_enabled', true)
            ->orderByRaw('FIELD(api_league_id, 39, 140, 135, 78, 61, 233, 307, 203)')
            ->get();

        return view('home', compact('liveMatches', 'todayMatches', 'yesterdayMatches', 'upcomingMatches', 'latestNews', 'leagues', 'standingsLeagues'));
    }
}
