<?php

namespace App\Jobs;

use App\Models\League;
use App\Services\FootballApiService;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Queue\Queueable;
use Illuminate\Support\Facades\Log;

class SyncLeagueStandingsJob implements ShouldQueue
{
    use Queueable;

    protected int $leagueId;

    protected int $season;

    /**
     * Create a new job instance.
     */
    public function __construct(int $leagueId, int $season)
    {
        $this->leagueId = $leagueId;
        $this->season = $season;
    }

    /**
     * Execute the job.
     */
    public function handle(FootballApiService $apiService): void
    {
        $league = League::where('api_league_id', $this->leagueId)->first();

        if (! $league || ! $league->is_enabled) {
            return;
        }

        $season = $this->season;
        $standings = $apiService->getStandings($this->leagueId, $season);

        // Fallback to previous seasons if current is not accessible (common with Free Plan)
        if (empty($standings)) {
            $fallbackSeasons = [$season - 1, $season - 2];
            foreach ($fallbackSeasons as $fs) {
                Log::info("Falling back to season $fs for league {$this->leagueId}");
                $standings = $apiService->getStandings($this->leagueId, $fs);
                if (! empty($standings)) {
                    Log::info("Successfully found standings for season $fs for league {$this->leagueId}");
                    break;
                }
            }
        }

        if (empty($standings)) {
            Log::warning("No standings found for league {$this->leagueId} in season {$this->season} or fallbacks");

            return;
        }

        $league->update(['standings_json' => $standings]);

        Log::info("Synced standings for league {$this->leagueId}");
    }
}
