<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;

class FootballApiService
{
    protected $baseUrl = 'https://v3.football.api-sports.io';

    protected $apiKey;

    public function __construct()
    {
        $this->apiKey = config('services.football_api.key');
    }

    protected function getHeaders()
    {
        return [
            'x-rapidapi-key' => $this->apiKey,
            'x-rapidapi-host' => 'v3.football.api-sports.io',
        ];
    }

    public function getLeagues()
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(60) // زيادة وقت الانتظار لـ 60 ثانية
            ->retry(3, 100) // إعادة المحاولة 3 مرات في حال الفشل
            ->get("{$this->baseUrl}/leagues");

        return $response->json()['response'] ?? [];
    }

    public function getTeams($leagueId, $season)
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(60)
            ->retry(3, 100)
            ->get("{$this->baseUrl}/teams", [
                'league' => $leagueId,
                'season' => $season,
            ]);

        return $response->json()['response'] ?? [];
    }

    public function getFixtures($params)
    {
        $queryParams = is_array($params) ? $params : ['date' => $params];

        $response = Http::withHeaders($this->getHeaders())
            ->timeout(60)
            ->retry(3, 100)
            ->get("{$this->baseUrl}/fixtures", $queryParams);

        return $response->json()['response'] ?? [];
    }

    public function getLeagueFixtures($leagueId, $season)
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(60)
            ->retry(3, 100)
            ->get("{$this->baseUrl}/fixtures", [
                'league' => $leagueId,
                'season' => $season,
            ]);

        return $response->json()['response'] ?? [];
    }

    public function getLiveFixtures()
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(60)
            ->retry(3, 100)
            ->get("{$this->baseUrl}/fixtures", [
                'live' => 'all',
            ]);

        return $response->json()['response'] ?? [];
    }

    public function getFixtureDetails($fixtureId)
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(60)
            ->retry(3, 100)
            ->get("{$this->baseUrl}/fixtures", [
                'id' => $fixtureId,
            ]);

        return $response->json()['response'][0] ?? null;
    }

    public function getStandings($leagueId, $season)
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(60)
            ->retry(3, 100)
            ->get("{$this->baseUrl}/standings", [
                'league' => $leagueId,
                'season' => $season,
            ]);

        return $response->json()['response'][0]['league']['standings'] ?? [];
    }

    public function getPlayer($playerId, $season)
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(60)
            ->retry(3, 100)
            ->get("{$this->baseUrl}/players", [
                'id' => $playerId,
                'season' => $season,
            ]);

        return $response->json()['response'][0] ?? null;
    }

    public function getPlayers($leagueId, $season, $page = 1)
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(60)
            ->retry(3, 100)
            ->get("{$this->baseUrl}/players", [
                'league' => $leagueId,
                'season' => $season,
                'page' => $page,
            ]);

        return $response->json() ?? [];
    }
}
