import 'package:flutter/material.dart';
import '../services/api_service.dart';
import '../models/football_match.dart';
import '../models/news.dart';
import 'news_detail_screen.dart';
import 'package:intl/intl.dart';

class HomeScreen extends StatefulWidget {
  @override
  _HomeScreenState createState() => _HomeScreenState();
}

class _HomeScreenState extends State<HomeScreen> {
  final ApiService _apiService = ApiService();
  
  @override
  Widget build(BuildContext context) {
    return Scaffold(
      body: CustomScrollView(
        slivers: [
          // Elegant Header
          SliverAppBar(
            floating: true,
            backgroundColor: Color(0xFF020617),
            title: Row(
              children: [
                Image.network('http://10.0.2.2/images/logo.png', height: 35, errorBuilder: (_, __, ___) => Icon(Icons.sports_soccer)),
                SizedBox(width: 10),
                Text('ScoreHub', style: TextStyle(fontWeight: FontWeight.black, letterSpacing: 1)),
              ],
            ),
            actions: [
              IconButton(icon: Icon(Icons.notifications_none_rounded), onPressed: () {}),
            ],
          ),

          // Live Matches Section
          SliverToBoxAdapter(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                _buildSectionHeader('مباريات مباشرة', Icons.live_tv, Colors.redAccent),
                FutureBuilder<List<FootballMatch>>(
                  future: _apiService.getLiveMatches(),
                  builder: (context, snapshot) {
                    if (!snapshot.hasData) return _buildLivePlaceholder();
                    return Container(
                      height: 170,
                      child: ListView.builder(
                        padding: EdgeInsets.symmetric(horizontal: 16),
                        scrollDirection: Axis.horizontal,
                        itemCount: snapshot.data!.length,
                        itemBuilder: (context, index) => _buildLiveMatchCard(snapshot.data![index]),
                      ),
                    );
                  },
                ),
              ],
            ),
          ),

          // Latest News Section
          SliverToBoxAdapter(
            child: _buildSectionHeader('أحدث الأخبار', Icons.newspaper, Colors.blueAccent),
          ),

          FutureBuilder<List<News>>(
            future: _apiService.getNews(),
            builder: (context, snapshot) {
              if (!snapshot.hasData) return SliverToBoxAdapter(child: Center(child: CircularProgressIndicator()));
              return SliverList(
                delegate: SliverChildBuilderDelegate(
                  (context, index) => _buildNewsItem(snapshot.data![index]),
                  childCount: snapshot.data!.length,
                ),
              );
            },
          ),
          
          SliverToBoxAdapter(child: SizedBox(height: 30)),
        ],
      ),
    );
  }

  Widget _buildSectionHeader(String title, IconData icon, Color color) {
    return Padding(
      padding: EdgeInsets.all(20),
      child: Row(
        children: [
          Icon(icon, size: 18, color: color),
          SizedBox(width: 10),
          Text(title, style: TextStyle(fontSize: 18, fontWeight: FontWeight.bold)),
          Spacer(),
          Text('عرض الكل', style: TextStyle(color: Colors.blueAccent, fontSize: 12)),
        ],
      ),
    );
  }

  Widget _buildLiveMatchCard(FootballMatch match) {
    return Container(
      width: 280,
      margin: EdgeInsets.only(left: 16),
      padding: EdgeInsets.all(16),
      decoration: BoxDecoration(
        gradient: LinearGradient(colors: [Color(0xFF1E293B), Color(0xFF0F172A)]),
        borderRadius: BorderRadius.circular(20),
        border: Border.all(color: Colors.redAccent.withOpacity(0.3)),
      ),
      child: Column(
        mainAxisAlignment: MainAxisAlignment.center,
        children: [
          Text(match.leagueName, style: TextStyle(fontSize: 10, color: Colors.grey)),
          SizedBox(height: 15),
          Row(
            mainAxisAlignment: MainAxisAlignment.spaceEvenly,
            children: [
              _teamSimple(match.homeTeamNameAr, match.homeTeamLogo),
              Column(
                children: [
                  Text('${match.homeScore} - ${match.awayScore}', style: TextStyle(fontSize: 22, fontWeight: FontWeight.black)),
                  Container(
                    padding: EdgeInsets.symmetric(horizontal: 6, vertical: 2),
                    decoration: BoxDecoration(color: Colors.redAccent, borderRadius: BorderRadius.circular(4)),
                    child: Text("LIVE ${match.status}'", style: TextStyle(fontSize: 9, fontWeight: FontWeight.bold)),
                  ),
                ],
              ),
              _teamSimple(match.awayTeamNameAr, match.awayTeamLogo),
            ],
          ),
        ],
      ),
    );
  }

  Widget _teamSimple(String name, String? logo) {
    return Column(
      children: [
        CircleAvatar(radius: 18, backgroundColor: Colors.white10, child: logo != null ? Image.network(logo, height: 20) : Icon(Icons.shield, size: 15)),
        SizedBox(height: 5),
        Container(width: 60, child: Text(name, textAlign: TextAlign.center, style: TextStyle(fontSize: 10, fontWeight: FontWeight.bold), overflow: TextOverflow.ellipsis)),
      ],
    );
  }

  Widget _buildNewsItem(News news) {
    return GestureDetector(
      onTap: () => Navigator.push(context, MaterialPageRoute(builder: (_) => NewsDetailScreen(news: news))),
      child: Container(
        margin: EdgeInsets.symmetric(horizontal: 20, vertical: 10),
        child: Row(
          children: [
            ClipRRect(
              borderRadius: BorderRadius.circular(12),
              child: Image.network('http://10.0.2.2/storage/${news.coverImage}', width: 100, height: 100, fit: BoxFit.cover, errorBuilder: (_,__,___) => Container(width:100, color: Colors.grey)),
            ),
            SizedBox(width: 15),
            Expanded(
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(news.titleAr, style: TextStyle(fontWeight: FontWeight.bold, fontSize: 15), maxLines: 2, overflow: TextOverflow.ellipsis),
                  SizedBox(height: 8),
                  Text(news.publishedAt != null ? DateFormat('yyyy/MM/dd').format(news.publishedAt!) : '', style: TextStyle(color: Colors.grey, fontSize: 11)),
                ],
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildLivePlaceholder() {
    return Container(height: 170, child: Center(child: Text('جاري جلب المباريات الحية...')));
  }
}
