import 'package:flutter/material.dart';
import '../services/api_service.dart';
import '../services/favorite_service.dart';
import '../models/football_match.dart';
import 'package:intl/intl.dart';

class MatchesScreen extends StatefulWidget {
  @override
  _MatchesScreenState createState() => _MatchesScreenState();
}

class _MatchesScreenState extends State<MatchesScreen> {
  final ApiService _apiService = ApiService();
  final FavoriteService _favoriteService = FavoriteService();
  late Future<List<FootballMatch>> _matchesFuture;
  List<int> _favorites = [];

  @override
  void initState() {
    super.initState();
    _loadData();
  }

  void _loadData() async {
    setState(() {
      _matchesFuture = _apiService.getMatches();
    });
    _favorites = await _favoriteService.getFavorites();
    if (mounted) setState(() {});
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: Text('المباريات', style: TextStyle(fontWeight: FontWeight.black)),
        centerTitle: true,
        backgroundColor: Color(0xFF0F172A),
        elevation: 0,
        actions: [
          IconButton(icon: Icon(Icons.refresh), onPressed: _loadData),
        ],
      ),
      body: FutureBuilder<List<FootballMatch>>(
        future: _matchesFuture,
        builder: (context, snapshot) {
          if (snapshot.connectionState == ConnectionState.waiting) {
            return Center(child: CircularProgressIndicator(color: Colors.blueAccent));
          } else if (snapshot.hasError) {
            return Center(child: Text('حدث خطأ في جلب البيانات'));
          } else if (!snapshot.hasData || snapshot.data!.isEmpty) {
            return Center(child: Text('لا توجد مباريات اليوم'));
          }

          final matches = snapshot.data!;
          return ListView.builder(
            padding: EdgeInsets.all(12),
            itemCount: matches.length,
            itemBuilder: (context, index) {
              final match = matches[index];
              return _buildMatchCard(match);
            },
          );
        },
      ),
    );
  }

  Widget _buildMatchCard(FootballMatch match) {
    bool isFav = _favorites.contains(match.id);
    return Container(
      margin: EdgeInsets.only(bottom: 16),
      decoration: BoxDecoration(
        color: Color(0xFF1E293B),
        borderRadius: BorderRadius.circular(16),
        border: Border.all(color: Colors.white.withOpacity(0.05)),
      ),
      child: Column(
        children: [
          // League Header & Favorite
          Container(
            padding: EdgeInsets.symmetric(horizontal: 16, vertical: 4),
            decoration: BoxDecoration(
              color: Colors.white.withOpacity(0.03),
              borderRadius: BorderRadius.vertical(top: Radius.circular(16)),
            ),
            child: Row(
              children: [
                Icon(Icons.emoji_events, size: 14, color: Colors.amber),
                SizedBox(width: 8),
                Text(
                  match.leagueName,
                  style: TextStyle(fontSize: 12, color: Colors.grey[400], fontWeight: FontWeight.bold),
                ),
                Spacer(),
                IconButton(
                  icon: Icon(
                    isFav ? Icons.star : Icons.star_border,
                    color: isFav ? Colors.amber : Colors.grey,
                    size: 20,
                  ),
                  onPressed: () async {
                    await _favoriteService.toggleFavorite(match.id);
                    _favorites = await _favoriteService.getFavorites();
                    setState(() {});
                  },
                ),
              ],
            ),
          ),
          
          Padding(
            padding: EdgeInsets.all(20),
            child: Row(
              mainAxisAlignment: MainAxisAlignment.spaceBetween,
              children: [
                // Home Team
                Expanded(
                  child: Column(
                    children: [
                      _buildTeamLogo(match.homeTeamLogo),
                      SizedBox(height: 10),
                      Text(
                        match.homeTeamNameAr,
                        textAlign: TextAlign.center,
                        style: TextStyle(fontSize: 14, fontWeight: FontWeight.bold),
                        maxLines: 1,
                        overflow: TextOverflow.ellipsis,
                      ),
                    ],
                  ),
                ),

                // Score / Status
                Container(
                  width: 100,
                  child: Column(
                    children: [
                      Text(
                        match.status == 'NS' 
                          ? DateFormat('HH:mm').format(match.matchDate)
                          : '${match.homeScore ?? 0} - ${match.awayScore ?? 0}',
                        style: TextStyle(
                          fontSize: 24, 
                          fontWeight: FontWeight.black,
                          color: _isLive(match.status) ? Colors.redAccent : Colors.white,
                        ),
                      ),
                      SizedBox(height: 4),
                      _buildStatusBadge(match.status),
                    ],
                  ),
                ),

                // Away Team
                Expanded(
                  child: Column(
                    children: [
                      _buildTeamLogo(match.awayTeamLogo),
                      SizedBox(height: 10),
                      Text(
                        match.awayTeamNameAr,
                        textAlign: TextAlign.center,
                        style: TextStyle(fontSize: 14, fontWeight: FontWeight.bold),
                        maxLines: 1,
                        overflow: TextOverflow.ellipsis,
                      ),
                    ],
                  ),
                ),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildTeamLogo(String? logo) {
    return Container(
      width: 50,
      height: 50,
      padding: EdgeInsets.all(4),
      decoration: BoxDecoration(
        color: Colors.white.withOpacity(0.05),
        shape: BoxShape.circle,
      ),
      child: logo != null && logo.startsWith('http')
          ? Image.network(logo, fit: BoxFit.contain)
          : Image.asset('assets/images/placeholder_team.png', fit: BoxFit.contain),
    );
  }

  Widget _buildStatusBadge(String status) {
    bool live = _isLive(status);
    return Container(
      padding: EdgeInsets.symmetric(horizontal: 10, vertical: 4),
      decoration: BoxDecoration(
        color: live ? Colors.redAccent.withOpacity(0.2) : Colors.black26,
        borderRadius: BorderRadius.circular(20),
        border: Border.all(color: live ? Colors.redAccent : Colors.transparent, width: 1),
      ),
      child: Text(
        _getStatusText(status),
        style: TextStyle(
          fontSize: 10, 
          color: live ? Colors.redAccent : Colors.grey,
          fontWeight: FontWeight.bold
        ),
      ),
    );
  }

  bool _isLive(String status) {
    return ['1H', 'HT', '2H', 'ET', 'BT', 'P', 'LIVE'].contains(status) || RegExp(r'^[0-9]+$').hasMatch(status);
  }

  String _getStatusText(String status) {
    if (status == 'NS') return 'لم تبدأ';
    if (status == 'FT') return 'انتهت';
    if (status == 'HT') return 'بين الشوطين';
    if (RegExp(r'^[0-9]+$').hasMatch(status)) return "'$status";
    return status;
  }
}
